<?php
session_start();

/**
 * email_confirm.php (PDO)
 * - التحقق من كود البريد من جدول email_code (غير مستخدم وغير منتهي)
 * - تفعيل الحساب فقط: users.is_active = 1
 * - لا يغيّر is_verified
 * - إنشاء device token (cookie) + user_token + سجلات:
 *   user_devices / user_sessions / user_logins
 * - يرجّع: user_id + email + device + user_token
 */

$frontendOrigin = 'https://yemencash.net';

/* ================== CORS ================== */
$reqOrigin = $_SERVER['HTTP_ORIGIN'] ?? '';
if ($reqOrigin === $frontendOrigin) {
  header("Access-Control-Allow-Origin: {$frontendOrigin}");
  header("Vary: Origin");
  header("Access-Control-Allow-Credentials: true");
}
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Max-Age: 600");
header("Content-Type: application/json; charset=UTF-8");
/* =================================================== */

// اجعلها false في الإنتاج حتى لا يظهر SQL للمستخدم
$DEBUG_MODE = false;
if ($DEBUG_MODE) { ini_set('display_errors', 1); error_reporting(E_ALL); }
else { ini_set('display_errors', 0); error_reporting(0); }

/* ================== Helpers ================== */
function respond(bool $success, string $message, array $extra = [], int $status = 200): void {
  http_response_code($status);
  echo json_encode(
    array_merge(['success' => $success, 'message' => $message], $extra),
    JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES
  );
  exit;
}

function read_json_input(): array {
  $raw = file_get_contents('php://input') ?: '';
  $data = json_decode($raw, true);
  if (!is_array($data)) {
    respond(false, 'JSON غير صالح.', [], 400);
  }
  return $data;
}

function log_error(Throwable $e): void {
  $dir = __DIR__ . '/../../logs';
  if (!is_dir($dir)) @mkdir($dir, 0755, true);
  $file = $dir . '/api_errors_' . date('Y-m-d') . '.log';
  $line = "[" . date('Y-m-d H:i:s') . "] " . $e->getMessage() . " | " . $e->getFile() . ":" . $e->getLine() . PHP_EOL;
  @file_put_contents($file, $line, FILE_APPEND | LOCK_EX);
}

/**
 * Device cookie (ثابت لكل جهاز)
 */
function getOrCreateDeviceToken(): string {
  $cookieName = 'yc_device';

  if (!empty($_COOKIE[$cookieName])) return (string)$_COOKIE[$cookieName];

  $token = hash('sha256', random_bytes(32));

  $options = [
    'expires'  => time() + (3600 * 24 * 365 * 2),
    'path'     => '/',
    // 'domain' => '.yemencash.net', // فعّلها فقط لو تحتاج مشاركة بين subdomains
    'secure'   => true,
    'httponly' => true,
    'samesite' => 'Lax',
  ];

  @setcookie($cookieName, $token, $options);
  return $token;
}
/* =================================================== */

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') {
  http_response_code(204);
  exit;
}

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  respond(false, 'استخدم POST فقط.', [], 405);
}

$input = read_json_input();

$email = trim((string)($input['email'] ?? ''));
$code  = preg_replace('/\D+/', '', (string)($input['code'] ?? ''));

if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
  respond(false, 'بريد غير صالح.', [], 400);
}
if ($code === '' || strlen($code) !== 6) {
  respond(false, 'كود غير صالح. يجب أن يكون 6 أرقام.', [], 400);
}

// DB
require_once __DIR__ . '/../config.php';
if (!isset($conn) || !($conn instanceof PDO)) {
  respond(false, 'اتصال قاعدة البيانات غير متوفر.', [], 500);
}

try {
  $conn->beginTransaction();

  // جلب الكود الأحدث المطابق (غير مستخدم وغير منتهي)
  $q = $conn->prepare("
    SELECT id, user_id, expires_at, used
    FROM email_code
    WHERE email = ? AND code = ?
    ORDER BY id DESC
    LIMIT 1
  ");
  $q->execute([$email, $code]);
  $row = $q->fetch(PDO::FETCH_ASSOC);

  if (!$row) {
    $conn->rollBack();
    respond(false, 'رمز التحقق غير صحيح.', [], 400);
  }

  if ((int)$row['used'] === 1) {
    $conn->rollBack();
    respond(false, 'تم استخدام هذا الرمز مسبقاً.', [], 400);
  }

  $expiresAt = (string)$row['expires_at'];
  if (strtotime($expiresAt) < time()) {
    $conn->rollBack();
    respond(false, 'انتهت صلاحية الرمز.', [], 400);
  }

  $userId = (int)($row['user_id'] ?? 0);
  if ($userId <= 0) {
    $conn->rollBack();
    respond(false, 'لا يمكن ربط الرمز بمستخدم.', [], 400);
  }

  // تأكد أن المستخدم موجود وأن البريد له
  $u0 = $conn->prepare("SELECT id, email, is_active FROM users WHERE id = ? AND email = ? LIMIT 1");
  $u0->execute([$userId, $email]);
  $userRow = $u0->fetch(PDO::FETCH_ASSOC);

  if (!$userRow) {
    $conn->rollBack();
    respond(false, 'تعذر العثور على المستخدم لهذا البريد.', [], 404);
  }

  // علّم الكود كمستخدم
  $u1 = $conn->prepare("UPDATE email_code SET used = 1 WHERE id = ? LIMIT 1");
  $u1->execute([(int)$row['id']]);

  // تفعيل الحساب (حتى لو كان مفعل سابقاً لا نعتبرها مشكلة)
  $u2 = $conn->prepare("UPDATE users SET is_active = 1 WHERE id = ? AND email = ? LIMIT 1");
  $u2->execute([$userId, $email]);

  // إنشاء توكنات + معلومات جهاز
  $device      = getOrCreateDeviceToken();
  $userToken   = hash('sha256', random_bytes(32));
  $ip          = $_SERVER['REMOTE_ADDR'] ?? null;
  $ua          = $_SERVER['HTTP_USER_AGENT'] ?? '';
  $deviceName  = mb_substr($ua, 0, 150);
  $deviceType  = 'Web Browser';
  $deviceModel = 'Browser';

  // user_devices upsert
  $d = $conn->prepare("
    INSERT INTO user_devices (user_id, device, device_name, device_type, device_model, created_at)
    VALUES (?, ?, ?, ?, ?, NOW())
    ON DUPLICATE KEY UPDATE
      user_id = VALUES(user_id),
      device_name = VALUES(device_name),
      device_type = VALUES(device_type),
      device_model = VALUES(device_model)
  ");
  $d->execute([$userId, $device, $deviceName, $deviceType, $deviceModel]);

  // user_sessions insert
  $s = $conn->prepare("
    INSERT INTO user_sessions
      (user_id, device, user_token, ip_address, country, governorate, device_name, device_type, device_model, session_start, last_activity_at, session_end, is_online)
    VALUES
      (?, ?, ?, ?, NULL, NULL, ?, ?, ?, NOW(), NOW(), NULL, 1)
  ");
  $s->execute([$userId, $device, $userToken, $ip, $deviceName, $deviceType, $deviceModel]);

  // user_logins insert
  $l = $conn->prepare("
    INSERT INTO user_logins
      (user_id, ip_address, country, governorate, device_name, device_type, device_model, device, user_token, login_at)
    VALUES
      (?, ?, NULL, NULL, ?, ?, ?, ?, ?, NOW())
  ");
  $l->execute([$userId, (string)$ip, $deviceName, $deviceType, $deviceModel, $device, $userToken]);

  $conn->commit();

  // جلسة PHP (اختياري)
  $_SESSION['user_id'] = $userId;

  respond(true, 'تم تفعيل الحساب بنجاح.', [
    'user_id'    => $userId,
    'email'      => $email,
    'device'     => $device,
    'user_token' => $userToken,
  ]);

} catch (Throwable $e) {
  if ($conn instanceof PDO && $conn->inTransaction()) $conn->rollBack();
  log_error($e);
  $msg = 'خطأ في الخادم';
  if ($DEBUG_MODE) $msg .= ' - ' . $e->getMessage();
  respond(false, $msg, [], 500);
}
