<?php
// get_balances.php (PDO) — GET/POST
// ✅ لا يستخدم wallets
// ✅ الرصيد = مجموع ledger_entries لكل أصل
// ✅ يرجع كل العملات المفعلة حتى لو رصيدها 0 (LEFT JOIN)
// ✅ يدعم: q, include_zero, hide_small + min_usd
// ✅ لا يستخدم client_id
// ✅ التوثيق: Session أو user_id + user_token (user_sessions.user_token)

if (session_status() === PHP_SESSION_NONE) session_start();

/* ================= CORS ================= */
$frontendOrigin = 'https://yemencash.net';
$reqOrigin = $_SERVER['HTTP_ORIGIN'] ?? '';
if ($reqOrigin === $frontendOrigin) {
  header("Access-Control-Allow-Origin: {$frontendOrigin}");
  header("Vary: Origin");
  header("Access-Control-Allow-Credentials: true");
}
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Max-Age: 600");
header("Content-Type: application/json; charset=UTF-8");

/* ================= DEBUG ================= */
$DEBUG_MODE = true;
if ($DEBUG_MODE) { ini_set('display_errors', 1); error_reporting(E_ALL); }
else { ini_set('display_errors', 0); error_reporting(0); }

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }

/* ================= Helpers ================= */
function respond(bool $success, string $message, array $extra = [], int $status = 200): void {
  http_response_code($status);
  echo json_encode(
    array_merge(['success' => $success, 'message' => $message], $extra),
    JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES
  );
  exit;
}

function log_error(Throwable $e): void {
  $dir = __DIR__ . '/../../logs';
  if (!is_dir($dir)) @mkdir($dir, 0755, true);
  $file = $dir . '/api_errors_' . date('Y-m-d') . '.log';
  $line = "[" . date('Y-m-d H:i:s') . "] " . $e->getMessage() . " | " . $e->getFile() . ":" . $e->getLine() . PHP_EOL;
  @file_put_contents($file, $line, FILE_APPEND | LOCK_EX);
}

function safe_float($v): float {
  $x = is_numeric($v) ? (float)$v : 0.0;
  return is_finite($x) ? $x : 0.0;
}

function safe_int($v, int $default = 0): int {
  if ($v === null || $v === '') return $default;
  return (int)$v;
}

function read_input(): array {
  $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';

  if ($method === 'POST') {
    $raw  = file_get_contents('php://input') ?: '';
    $json = json_decode($raw, true);
    if (is_array($json)) return $json;
    return $_POST ?? [];
  }

  return $_GET ?? [];
}

/* ================= DB ================= */
require_once __DIR__ . '/../config.php';
if (!isset($conn) || !($conn instanceof PDO)) {
  respond(false, 'اتصال قاعدة البيانات غير متوفر.', [], 500);
}

/* ================= AUTH: Session أو user_id+user_token ================= */
function resolveUserId(PDO $conn, array $in): int {
  // 1) Session
  $sid = isset($_SESSION['user_id']) ? (int)$_SESSION['user_id'] : 0;
  if ($sid > 0) return $sid;

  // 2) user_id + user_token (من GET أو POST JSON)
  $uid = isset($in['user_id']) ? (int)$in['user_id'] : 0;
  $tok = trim((string)($in['user_token'] ?? ''));
  if ($uid <= 0 || $tok === '') return 0;

  // ✅ العمود الصحيح: user_token
  $st = $conn->prepare("
    SELECT user_id
    FROM user_sessions
    WHERE user_id = :uid
      AND user_token = :tok
      AND is_online = 1
    LIMIT 1
  ");
  $st->execute([':uid' => $uid, ':tok' => $tok]);
  $row = $st->fetch(PDO::FETCH_ASSOC);

  return $row ? (int)$row['user_id'] : 0;
}

try {
  $in = read_input();

  $userId = resolveUserId($conn, $in);
  if ($userId <= 0) {
    respond(false, 'غير مصرح. الرجاء تسجيل الدخول.', ['unauthorized' => true], 401);
  }

  /* ================= Params ================= */
  $q           = trim((string)($in['q'] ?? ''));
  $includeZero = safe_int($in['include_zero'] ?? 1, 1);   // 0 يخفي الصفري
  $hideSmall   = safe_int($in['hide_small'] ?? 0, 0);     // 1 يخفي الصغير
  $minUsd      = safe_float($in['min_usd'] ?? 1);         // حد الصغير

  /* ================= Query ================= */
  $whereAssets = "WHERE a.is_active = 1";
  $params = [':uid' => $userId];

  if ($q !== '') {
    $whereAssets .= " AND (a.code LIKE :q OR a.name_ar LIKE :q)";
    $params[':q'] = '%' . $q . '%';
  }

  // نكرر SUM داخل total_amount لتفادي مشكلة ORDER BY على aliases
  $sql = "
    SELECT
      a.id         AS asset_id,
      a.code       AS code,
      a.name_ar    AS name_ar,
      a.type       AS type,
      a.decimals   AS decimals,
      a.icon_url   AS icon_url,
      a.sort_order AS sort_order,

      COALESCE(SUM(
        CASE
          WHEN le.balance_type = 'available' AND le.status = 'posted' THEN le.amount
          ELSE 0
        END
      ), 0) AS available,

      COALESCE(SUM(
        CASE
          WHEN le.balance_type = 'locked' AND le.status = 'posted' THEN le.amount
          ELSE 0
        END
      ), 0) AS locked,

      (
        COALESCE(SUM(
          CASE
            WHEN le.balance_type = 'available' AND le.status = 'posted' THEN le.amount
            ELSE 0
          END
        ), 0)
        +
        COALESCE(SUM(
          CASE
            WHEN le.balance_type = 'locked' AND le.status = 'posted' THEN le.amount
            ELSE 0
          END
        ), 0)
      ) AS total_amount

    FROM assets a
    LEFT JOIN ledger_entries le
      ON le.user_id  = :uid
     AND le.asset_id = a.id

    {$whereAssets}
    GROUP BY a.id, a.code, a.name_ar, a.type, a.decimals, a.icon_url, a.sort_order
    ORDER BY total_amount DESC, a.sort_order DESC, a.code ASC
  ";

  $stmt = $conn->prepare($sql);
  $stmt->execute($params);
  $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

  /* ================= Build response ================= */
  $items = [];
  $totalUsd = 0.0;

  foreach ($rows as $r) {
    $code = strtoupper((string)($r['code'] ?? ''));

    $available = safe_float($r['available'] ?? 0);
    $locked    = safe_float($r['locked'] ?? 0);
    $total     = safe_float($r['total_amount'] ?? ($available + $locked));

    if ((int)$includeZero === 0 && $total == 0.0) continue;

    // USD value (حاليًا: stable فقط كبداية)
    $usdPrice = null;
    if (in_array($code, ['USD', 'USDT', 'USDC'], true)) $usdPrice = 1.0;
    $usdValue = ($usdPrice !== null) ? ($total * $usdPrice) : null;

    if ((int)$hideSmall === 1 && $usdValue !== null && $usdValue < $minUsd) continue;
    if ($usdValue !== null) $totalUsd += $usdValue;

    $items[] = [
      'asset_id'  => (int)($r['asset_id'] ?? 0),
      'code'      => $code,
      'name_ar'   => (string)($r['name_ar'] ?? ''),
      'type'      => (string)($r['type'] ?? ''),
      'decimals'  => (int)($r['decimals'] ?? 2),
      'icon_url'  => $r['icon_url'] ?? null,

      'available' => (string)$available,
      'locked'    => (string)$locked,
      'total'     => (string)$total,

      'usd_value' => $usdValue,
    ];
  }

  respond(true, 'تم جلب الأرصدة بنجاح.', [
    'user_id'   => $userId,
    'total_usd' => round($totalUsd, 2),
    'count'     => count($items),
    'items'     => $items,
  ]);

} catch (Throwable $e) {
  log_error($e);
  $msg = 'خطأ في الخادم';
  if ($DEBUG_MODE) $msg .= ' - ' . $e->getMessage();
  respond(false, $msg, [], 500);
}
