<?php
session_start();

/**
 * password_reset_request.php (PDO) - mail() only + reCAPTCHA (FINAL)
 * ✅ يستقبل البريد الإلكتروني ويُنشئ كود إعادة تعيين (TTL=60s)
 * ✅ يغلق الأكواد السابقة (used=1)
 * ✅ يرسل الكود عبر mail() فقط
 * ✅ يتحقق من Google reCAPTCHA Server-side
 *
 * الواجهة ترسل JSON:
 * { "email": "...", "recaptcha_token": "TOKEN" }
 */

$frontendOrigin = 'https://eazzybit.com';
$reqOrigin = $_SERVER['HTTP_ORIGIN'] ?? '';
if ($reqOrigin === $frontendOrigin) {
  header("Access-Control-Allow-Origin: {$frontendOrigin}");
  header("Vary: Origin");
  header("Access-Control-Allow-Credentials: true");
}
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Max-Age: 600");
header("Content-Type: application/json; charset=UTF-8");

// وضع تطوير
$DEBUG_MODE = true;
if ($DEBUG_MODE) { ini_set('display_errors', 1); error_reporting(E_ALL); }
else { ini_set('display_errors', 0); error_reporting(0); }

// Preflight
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') {
  http_response_code(204);
  exit;
}

// POST فقط
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['success' => false, 'message' => 'طريقة الطلب غير مسموحة. استخدم POST فقط.'], JSON_UNESCAPED_UNICODE);
  exit;
}

/* ================= أدوات مساعدة ================= */
function respond(bool $success, string $message, array $extra = [], int $status = 200): void {
  http_response_code($status);
  echo json_encode(
    array_merge(['success' => $success, 'message' => $message], $extra),
    JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES
  );
  exit;
}

function read_json_input(): array {
  $raw = file_get_contents('php://input') ?: '';
  $data = json_decode($raw, true);
  if (!is_array($data)) {
    respond(false, 'البيانات المرسلة غير صالحة', [], 400);
  }
  return $data;
}

function log_error(Throwable $e): void {
  $dir = __DIR__ . '/../../logs';
  if (!is_dir($dir)) @mkdir($dir, 0755, true);
  $file = $dir . '/api_errors_' . date('Y-m-d') . '.log';
  $line = "[" . date('Y-m-d H:i:s') . "] " . $e->getMessage() . " | " . $e->getFile() . ":" . $e->getLine() . PHP_EOL;
  @file_put_contents($file, $line, FILE_APPEND | LOCK_EX);
}

/**
 * ✅ mail() only (مُحسّن للتوافق)
 */
function sendResetCodeMail(string $email, string $code, string $expiresAt): bool {
  $fromEmail = "no-reply@eazzybit.com";
  $fromName  = "EazzyBit";

  $subject = "رمز إعادة تعيين كلمة المرور في EazzyBit";
  if (function_exists('mb_encode_mimeheader')) {
    $subject = mb_encode_mimeheader($subject, "UTF-8", "B", "\r\n");
  }

  $body =
    "مرحباً\r\n\r\n" .
    "رمز إعادة تعيين كلمة المرور هو:\r\n" .
    "{$code}\r\n\r\n" .
    "صالح حتى: {$expiresAt}\r\n\r\n" .
    "إذا لم تطلب إعادة تعيين كلمة المرور، تجاهل هذه الرسالة.\r\n\r\n" .
    "فريق EazzyBit\r\n";

  $body = wordwrap($body, 70, "\r\n");

  @ini_set('sendmail_from', $fromEmail);

  $headers = [];
  $headers[] = "MIME-Version: 1.0";
  $headers[] = "Content-Type: text/plain; charset=UTF-8";
  $headers[] = "Content-Transfer-Encoding: 8bit";
  $headers[] = "From: {$fromName} <{$fromEmail}>";
  $headers[] = "Reply-To: {$fromEmail}";
  $headers[] = "X-Mailer: PHP/" . phpversion();

  $headersStr = implode("\r\n", $headers);

  // envelope sender (قد يفشل في بعض الاستضافات)
  $params = "-f{$fromEmail}";
  $ok = @mail($email, $subject, $body, $headersStr, $params);
  if (!$ok) {
    $ok = @mail($email, $subject, $body, $headersStr);
  }
  return (bool)$ok;
}

/* ================= reCAPTCHA (Server-side verify) ================= */

function verify_recaptcha(string $token, string $remoteIp = ''): array {
  if (!defined('RECAPTCHA_SECRET') || !RECAPTCHA_SECRET) {
    return ['ok' => false, 'reason' => 'RECAPTCHA_SECRET_NOT_SET'];
  }

  $url = 'https://www.google.com/recaptcha/api/siteverify';

  $post = [
    'secret'   => RECAPTCHA_SECRET,
    'response' => $token,
  ];
  if ($remoteIp !== '') $post['remoteip'] = $remoteIp;

  $respRaw = null;

  // الأفضل: cURL
  if (function_exists('curl_init')) {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
      CURLOPT_POST => true,
      CURLOPT_POSTFIELDS => http_build_query($post),
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_CONNECTTIMEOUT => 6,
      CURLOPT_TIMEOUT => 10,
    ]);
    $respRaw = curl_exec($ch);
    curl_close($ch);
  } else {
    // بديل: file_get_contents
    $context = stream_context_create([
      'http' => [
        'method'  => 'POST',
        'header'  => "Content-Type: application/x-www-form-urlencoded\r\n",
        'content' => http_build_query($post),
        'timeout' => 10,
      ]
    ]);
    $respRaw = @file_get_contents($url, false, $context);
  }

  if (!$respRaw) return ['ok' => false, 'reason' => 'VERIFY_REQUEST_FAILED'];

  $json = json_decode($respRaw, true);
  if (!is_array($json)) return ['ok' => false, 'reason' => 'VERIFY_BAD_RESPONSE'];

  if (empty($json['success'])) {
    return ['ok' => false, 'reason' => 'VERIFY_NOT_SUCCESS', 'errors' => $json['error-codes'] ?? []];
  }

  // v3 score (اختياري)
  if (isset($json['score']) && defined('RECAPTCHA_MIN_SCORE')) {
    $min = (float)RECAPTCHA_MIN_SCORE;
    $score = (float)$json['score'];
    if ($score < $min) return ['ok' => false, 'reason' => 'LOW_SCORE', 'score' => $score];
  }

  return ['ok' => true, 'data' => $json];
}

/* ================= تشغيل ================= */
require_once __DIR__ . '/../config.php';
if (!isset($conn) || !($conn instanceof PDO)) {
  respond(false, 'اتصال قاعدة البيانات غير متوفر.', [], 500);
}

try {
  $input = read_json_input();

  $email = strtolower(trim((string)($input['email'] ?? '')));

  $recaptchaToken =
    trim((string)($input['recaptcha_token'] ?? '')) ?:
    trim((string)($input['g_recaptcha_response'] ?? '')) ?:
    trim((string)($input['g-recaptcha-response'] ?? ''));

  if ($email === '') {
    respond(false, 'البريد الإلكتروني مطلوب', [], 400);
  }
  if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    respond(false, 'البريد الإلكتروني غير صالح', [], 400);
  }

  // ✅ reCAPTCHA إلزامي
  if ($recaptchaToken === '') {
    respond(false, 'التحقق مطلوب. أعد المحاولة.', ['captcha' => 'required'], 403);
  }

  $remoteIp = $_SERVER['REMOTE_ADDR'] ?? '';
  $cap = verify_recaptcha($recaptchaToken, $remoteIp);
  if (!$cap['ok']) {
    $extra = ['captcha' => 'failed'];
    if ($DEBUG_MODE) $extra['captcha_reason'] = $cap['reason'] ?? 'unknown';
    respond(false, 'فشل التحقق. حاول مرة أخرى.', $extra, 403);
  }

  // ✅ تأكد من وجود المستخدم
  $stmt = $conn->prepare("SELECT id FROM users WHERE email = ? LIMIT 1");
  $stmt->execute([$email]);
  $u = $stmt->fetch(PDO::FETCH_ASSOC);

  if (!$u || empty($u['id'])) {
    respond(false, 'هذا البريد غير مسجل لدينا.', ['not_found' => true], 404);
  }

  $userId = (int)$u['id'];

  // ✅ إنشاء كود
  $ttlSeconds = 60;
  $code = (string)random_int(100000, 999999);
  $expiresAt = date('Y-m-d H:i:s', time() + $ttlSeconds);

  $conn->beginTransaction();

  // إغلاق الأكواد السابقة
  $conn->prepare("UPDATE password_reset_code SET used = 1 WHERE user_id = ? AND used = 0")
       ->execute([$userId]);

  // إدراج كود جديد
  $ins = $conn->prepare("
    INSERT INTO password_reset_code (email, user_id, code, used, expires_at, created_at)
    VALUES (?, ?, ?, 0, ?, NOW())
  ");
  $ins->execute([$email, $userId, $code, $expiresAt]);

  // إرسال البريد
  $sent = sendResetCodeMail($email, $code, $expiresAt);
  if (!$sent) {
    $conn->rollBack();
    respond(false, 'تعذّر إرسال رسالة البريد. حاول لاحقاً.', [], 500);
  }

  $conn->commit();

  respond(true, 'تم إرسال رمز إعادة التعيين إلى بريدك الإلكتروني.', [
    'email'      => $email,
    'user_id'    => $userId,
    'expires_at' => $expiresAt,
    'ttl'        => $ttlSeconds
  ]);

} catch (PDOException $e) {
  if ($conn instanceof PDO && $conn->inTransaction()) $conn->rollBack();
  log_error($e);

  $msg = 'خطأ في الخادم';
  if ($DEBUG_MODE) $msg .= ' - ' . $e->getMessage();
  respond(false, $msg, [], 500);

} catch (Throwable $e) {
  if ($conn instanceof PDO && $conn->inTransaction()) $conn->rollBack();
  log_error($e);

  $msg = 'خطأ في الخادم';
  if ($DEBUG_MODE) $msg .= ' - ' . $e->getMessage();
  respond(false, $msg, [], 500);
}
