<?php
session_start();

/**
 * email_request.php (FINAL)
 * - يولّد كود تفعيل ويرسله للحساب غير المفعل فقط (users.is_active = 0)
 * - يخزّن الكود في email_code (used=0 + expires_at)
 * - Rate limit: يمنع الإرسال المتكرر خلال 60 ثانية
 * - لا يغيّر is_verified
 * - يقبل JSON:
 *   { "email": "example@gmail.com" }
 *   أو { "action": "resend_code", "email": "example@gmail.com" }
 *
 * ملاحظة: هذا الملف يستخدم أيضاً كـ "مصدر واحد" لمنطق إرسال الكود،
 * ويمكن استدعاؤه من get_register.php بوضع LIB mode عبر تعريف YC_EMAIL_REQUEST_LIB.
 */

$frontendOrigin = 'https://yemencash.net';

/* ================== Shared Helpers (لا تعتمد على Respond) ================== */
function yc_log_error(Throwable $e): void {
  $dir = __DIR__ . '/../../logs';
  if (!is_dir($dir)) @mkdir($dir, 0755, true);
  $file = $dir . '/api_errors_' . date('Y-m-d') . '.log';
  $line = "[" . date('Y-m-d H:i:s') . "] " . $e->getMessage() . " | " . $e->getFile() . ":" . $e->getLine() . PHP_EOL;
  @file_put_contents($file, $line, FILE_APPEND | LOCK_EX);
}

function yc_sendMailCode(string $email, string $name, string $code, string $expiresAt): bool {
  $subject = "رمز تفعيل بريدك في YemenCash";
  $body = "مرحباً " . ($name !== '' ? $name : 'عزيزنا المستخدم') . "\n\n"
        . "رمز التفعيل الخاص بحسابك هو:\n"
        . $code . "\n\n"
        . "صالح حتى: " . $expiresAt . " (لمدة دقيقة واحدة).\n\n"
        . "فريق YemenCash";

  $headers  = "MIME-Version: 1.0\r\n";
  $headers .= "Content-Type: text/plain; charset=UTF-8\r\n";
  $headers .= "From: YemenCash <no-reply@yemencash.net>\r\n";

  return (bool) @mail($email, $subject, $body, $headers);
}

/**
 * المصدر الواحد لإرسال/إعادة إرسال كود البريد
 * @return array ['success'=>bool,'message'=>string,'status'=>int,'extra'=>array]
 */
function yc_email_request_send_code(PDO $conn, int $userId, string $email, string $name, int $ttlSeconds = 60, bool $applyRateLimit = true): array {
  try {
    // Rate limit: آخر إرسال خلال ttl
    if ($applyRateLimit) {
      $lastQ = $conn->prepare("
        SELECT created_at
        FROM email_code
        WHERE email = ? AND user_id = ?
        ORDER BY id DESC
        LIMIT 1
      ");
      $lastQ->execute([$email, $userId]);
      $last = $lastQ->fetch(PDO::FETCH_ASSOC);

      if ($last && !empty($last['created_at'])) {
        $diff = time() - strtotime((string)$last['created_at']);
        if ($diff < $ttlSeconds) {
          $remain = $ttlSeconds - $diff;
          return [
            'success' => false,
            'message' => "تم إرسال رمز مؤخراً. حاول بعد {$remain} ثانية.",
            'status'  => 429,
            'extra'   => ['retry_after' => $remain]
          ];
        }
      }
    }

    $code      = (string) random_int(100000, 999999);
    $expiresAt = date('Y-m-d H:i:s', time() + $ttlSeconds);

    $startedTx = false;
    if (!$conn->inTransaction()) {
      $conn->beginTransaction();
      $startedTx = true;
    }

    // اغلاق الأكواد السابقة غير المستخدمة
    $conn->prepare("UPDATE email_code SET used = 1 WHERE user_id = ? AND used = 0")->execute([$userId]);

    // إدخال كود جديد
    $ins = $conn->prepare("
      INSERT INTO email_code (email, user_id, code, used, expires_at, created_at)
      VALUES (?, ?, ?, 0, ?, NOW())
    ");
    $ins->execute([$email, $userId, $code, $expiresAt]);

    // إرسال البريد
    $sent = yc_sendMailCode($email, $name, $code, $expiresAt);
    if (!$sent) {
      if ($conn->inTransaction()) $conn->rollBack();
      return [
        'success' => false,
        'message' => 'تم إنشاء رمز التفعيل ولكن تعذّر إرسال البريد. حاول لاحقاً.',
        'status'  => 500,
        'extra'   => []
      ];
    }

    if ($startedTx) $conn->commit();

    return [
      'success' => true,
      'message' => 'تم إرسال رمز التفعيل إلى بريدك الإلكتروني.',
      'status'  => 200,
      'extra'   => [
        'email'      => $email,
        'expires_at' => $expiresAt,
        'ttl'        => $ttlSeconds
      ]
    ];

  } catch (Throwable $e) {
    if ($conn->inTransaction()) $conn->rollBack();
    yc_log_error($e);
    return [
      'success' => false,
      'message' => 'خطأ في الخادم',
      'status'  => 500,
      'extra'   => []
    ];
  }
}
/* =================================================== */


/* ================== Endpoint Mode ================== */
if (!defined('YC_EMAIL_REQUEST_LIB')) {

  /* ================== CORS ================== */
  $reqOrigin = $_SERVER['HTTP_ORIGIN'] ?? '';
  if ($reqOrigin === $frontendOrigin) {
    header("Access-Control-Allow-Origin: {$frontendOrigin}");
    header("Vary: Origin");
    header("Access-Control-Allow-Credentials: true");
  }
  header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");
  header("Access-Control-Allow-Methods: POST, OPTIONS");
  header("Access-Control-Max-Age: 600");
  header("Content-Type: application/json; charset=UTF-8");
  /* =================================================== */

  $DEBUG_MODE = true;
  if ($DEBUG_MODE) { ini_set('display_errors', 1); error_reporting(E_ALL); }
  else { ini_set('display_errors', 0); error_reporting(0); }

  function respond(bool $success, string $message, array $extra = [], int $status = 200): void {
    http_response_code($status);
    echo json_encode(
      array_merge(['success' => $success, 'message' => $message], $extra),
      JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES
    );
    exit;
  }

  function read_json_input(): array {
    $raw = file_get_contents('php://input') ?: '';
    $data = json_decode($raw, true);
    if (!is_array($data)) {
      respond(false, 'JSON غير صالح.', [], 400);
    }
    return $data;
  }

  if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') {
    http_response_code(204);
    exit;
  }

  if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
    respond(false, 'استخدم POST فقط.', [], 405);
  }

  $input  = read_json_input();
  $action = trim((string)($input['action'] ?? 'request')); // للتوافق فقط
  $email  = trim((string)($input['email'] ?? ''));

  if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
    respond(false, 'بريد غير صالح.', [], 400);
  }

  // DB
  require_once __DIR__ . '/../config.php';
  if (!isset($conn) || !($conn instanceof PDO)) {
    respond(false, 'اتصال قاعدة البيانات غير متوفر.', [], 500);
  }

  try {
    $ttlSeconds = 60;

    // جلب المستخدم
    $u = $conn->prepare("SELECT id, name, is_active FROM users WHERE email = ? LIMIT 1");
    $u->execute([$email]);
    $user = $u->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
      respond(false, 'البريد غير مسجل.', ['exists' => false], 404);
    }

    $userId = (int)($user['id'] ?? 0);
    $name   = (string)($user['name'] ?? '');
    $active = (int)($user['is_active'] ?? 0);

    if ($userId <= 0) {
      respond(false, 'تعذر تحديد المستخدم.', [], 500);
    }

    if ($active === 1) {
      respond(false, 'الحساب مفعل مسبقاً. استخدم تسجيل الدخول.', ['exists' => true, 'is_active' => 1], 409);
    }

    $r = yc_email_request_send_code($conn, $userId, $email, $name, $ttlSeconds, true);
    respond((bool)$r['success'], (string)$r['message'], (array)$r['extra'], (int)$r['status']);

  } catch (Throwable $e) {
    yc_log_error($e);
    $msg = 'خطأ في الخادم';
    if ($DEBUG_MODE) $msg .= ' - ' . $e->getMessage();
    respond(false, $msg, [], 500);
  }
}
