<?php
session_start();

/**
 * email_request.php (EazzyBit - FINAL | Banner + Dark/Grey Design | No Banner Border | No Banner Corner Crop)
 * - يولّد كود تفعيل ويرسله للحساب غير المفعل فقط (users.is_active = 0)
 * - يخزّن الكود في email_code (used=0 + expires_at)
 * - Rate limit: يمنع الإرسال المتكرر خلال 60 ثانية (retry_after)
 * - لا يغيّر is_verified ولا is_active
 * - يقبل JSON:
 *   { "email": "example@gmail.com" }
 *   أو { "action": "resend_code", "email": "example@gmail.com" } (للتوافق)
 *
 * ✅ يدعم LIB mode:
 *   define('YC_EMAIL_REQUEST_LIB', true); require 'email_request.php';
 *   ثم استدعاء: yc_email_request_send_code(...)
 */

$frontendOrigin = 'https://eazzybit.com';

/* ================== Brand / Template Settings ==================
 * مهم: لازم يكون رابط البانر HTTPS مباشر حتى يظهر داخل Gmail
 */
if (!defined('EB_BRAND_NAME'))  define('EB_BRAND_NAME', 'EazzyBit');
if (!defined('EB_SITE_URL'))    define('EB_SITE_URL', 'https://www.eazzybit.com');
if (!defined('EB_FROM_EMAIL'))  define('EB_FROM_EMAIL', 'no-reply@eazzybit.com');
if (!defined('EB_FROM_NAME'))   define('EB_FROM_NAME', 'EazzyBit');
if (!defined('EB_BANNER_URL'))  define('EB_BANNER_URL', 'https://eazzybit.com/banner.png'); // <-- غيّره لرابط بانرك الفعلي

/* ================== Shared Helpers (لا تعتمد على respond) ================== */
function yc_log_error(Throwable $e): void {
  $dir = __DIR__ . '/../../logs';
  if (!is_dir($dir)) @mkdir($dir, 0755, true);
  $file = $dir . '/api_errors_' . date('Y-m-d') . '.log';
  $line = "[" . date('Y-m-d H:i:s') . "] " . $e->getMessage() . " | " . $e->getFile() . ":" . $e->getLine() . PHP_EOL;
  @file_put_contents($file, $line, FILE_APPEND | LOCK_EX);
}

function yc_format_expires_human(string $expiresAtDb): string {
  $ts = strtotime($expiresAtDb);
  if (!$ts) return $expiresAtDb;
  // نفس الشكل: 05:12:41 21-12-2025
  return date('H:i:s d-m-Y', $ts);
}

function yc_build_email_plain(string $name, string $code, string $expiresAtDb, int $ttlSeconds): string {
  $safeName = trim($name) !== '' ? trim($name) : 'عزيزنا المستخدم';
  $expiresHuman = yc_format_expires_human($expiresAtDb);

  return
    "مرحباً {$safeName}\r\n\r\n" .
    "كود التحقق لإتمام: تأكيد البريد\r\n" .
    "استخدم الكود التالي:\r\n" .
    "{$code}\r\n\r\n" .
    "الكود صالح حتى: {$expiresHuman} (مدة {$ttlSeconds} ثانية)\r\n\r\n" .
    "إذا لم تطلب هذا الكود، تجاهل هذه الرسالة.\r\n\r\n" .
    "مع أطيب تمنياتنا لك بالنجاح المستمر\r\n" .
    "فريق " . EB_BRAND_NAME . "\r\n";
}

/**
 * تصميم داكن/رمادي مطابق للصورة + معالجة مشاكل:
 * - بدون أي حدود حول البانر (no border/outline/gaps)
 * - بدون قص زوايا البانر (إلغاء border-radius/overflow في الحاوية العلوية)
 * - تقليل الفراغات قدر الإمكان (Gmail قد يضيف هوامش خارجية لا يمكن منعها 100%)
 */
function yc_build_email_html(string $email, string $name, string $code, string $expiresAtDb, int $ttlSeconds): string {
  $expiresHuman = yc_format_expires_human($expiresAtDb);

  $brandName = EB_BRAND_NAME;
  $bannerUrl = EB_BANNER_URL;

  // ألوان مطابقة تقريباً للصورة
  $bgOuter   = "#2c2f33";
  $wrapBg    = "#2c2f33";
  $sectionA  = "#3b4046";
  $sectionB  = "#2c2f33";
  $gold      = "#b7923a";
  $white     = "#ffffff";
  $soft      = "#cfd2d6";
  $divider   = "#3a3f45"; // بدل rgba لتوافق Outlook/Gmail
  $year      = date('Y');

  return '<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <meta name="color-scheme" content="dark light" />
  <meta name="supported-color-schemes" content="dark light" />
  <title>كود التحقق - '.$brandName.'</title>
</head>
<body style="margin:0;padding:0;background:'.$bgOuter.';font-family:Tahoma,Arial,sans-serif;">
  <div style="display:none;max-height:0;overflow:hidden;opacity:0;color:transparent;">
    كود التحقق لتأكيد البريد في '.$brandName.'
  </div>

  <table role="presentation" width="100%" cellpadding="0" cellspacing="0" border="0" bgcolor="'.$bgOuter.'"
         style="width:100%;background:'.$bgOuter.';margin:0;padding:0;border-spacing:0;border-collapse:collapse;mso-table-lspace:0pt;mso-table-rspace:0pt;">
    <tr>
      <td align="center" valign="top" style="margin:0;padding:0;">

        <table role="presentation" width="600" cellpadding="0" cellspacing="0" border="0" bgcolor="'.$wrapBg.'"
               style="width:100%;max-width:600px;background:'.$wrapBg.';margin:0 auto;border-spacing:0;border-collapse:collapse;mso-table-lspace:0pt;mso-table-rspace:0pt;">

          <!-- Banner (بدون حدود/فراغات/قص) -->
          <tr>
            <td bgcolor="'.$wrapBg.'" style="padding:0;margin:0;line-height:0;font-size:0;background:'.$wrapBg.';">
              <img src="'.$bannerUrl.'" alt="'.$brandName.'"
                   width="600" border="0"
                   style="display:block;width:100%;max-width:600px;height:auto;border:0;outline:none;text-decoration:none;margin:0;padding:0;line-height:0;font-size:0;-ms-interpolation-mode:bicubic;" />
            </td>
          </tr>

          <!-- Section A -->
          <tr>
            <td bgcolor="'.$sectionA.'" style="background:'.$sectionA.';padding:22px 18px;text-align:center;">
              <div style="color:'.$gold.';font-size:26px;font-weight:800;line-height:1.35;">
                كود التحقق لإتمام: تأكيد البريد
              </div>

              <div style="margin-top:12px;color:'.$white.';font-size:20px;font-weight:700;">
                استخدم الكود التالي
              </div>

              <div style="margin-top:14px;font-size:54px;font-weight:900;letter-spacing:10px;color:'.$white.';">
                '.$code.'
              </div>

              <div style="margin-top:12px;color:'.$white.';font-size:18px;">
                لإتمام: تأكيد البريد
              </div>
            </td>
          </tr>

          <!-- Divider -->
          <tr>
            <td height="1" bgcolor="'.$divider.'" style="height:1px;background:'.$divider.';font-size:0;line-height:0;padding:0;margin:0;"></td>
          </tr>

          <!-- Section B -->
          <tr>
            <td bgcolor="'.$sectionB.'" style="background:'.$sectionB.';padding:18px 18px;text-align:center;">
              <div style="color:'.$soft.';font-size:16px;line-height:2.0;">
                يصلك هذا البريد الإلكتروني لأنك طلبت كود تحقق لتأكيد بريدك
                في <b style="color:'.$white.';">'.$brandName.'</b>.
                <br/>
                إذا لم تكن أنت من طلب ذلك، يمكنك تجاهل الرسالة.
              </div>

              <div style="margin-top:14px;color:'.$soft.';font-size:16px;line-height:2.0;">
                الكود صالح حتى:
                <b style="color:'.$white.';">'.$expiresHuman.'</b>
                <span style="color:'.$soft.';">(مدة '.$ttlSeconds.' ثانية)</span>
              </div>

              <div style="margin-top:16px;color:'.$soft.';font-size:16px;line-height:2.0;">
                إذا كان لديك أي أسئلة أو استفسارات، لا تتردد في التواصل معنا.
                <br/>نحن هنا لمساعدتك.
              </div>

              <div style="margin-top:20px;color:'.$soft.';font-size:16px;">
                مع أطيب تمنياتنا لك بالنجاح المستمر
              </div>

              <div style="margin-top:10px;color:'.$gold.';font-size:18px;font-weight:900;">
                فريق '.$brandName.'
              </div>
            </td>
          </tr>

          <!-- Footer -->
          <tr>
            <td bgcolor="'.$sectionB.'" style="background:'.$sectionB.';padding:12px 18px;text-align:center;border-top:1px solid '.$divider.';">
              <div style="color:rgba(255,255,255,.55);font-size:13px;line-height:1.7;">
                هذه رسالة تلقائية، يرجى عدم الرد عليها.
              </div>
              <div style="margin-top:8px;color:rgba(255,255,255,.45);font-size:13px;line-height:1.7;">
                '.$year.' - جميع الحقوق محفوظة لـ '.$brandName.'
              </div>
            </td>
          </tr>

        </table>

      </td>
    </tr>
  </table>
</body>
</html>';
}

/**
 * ✅ mail() only + multipart/alternative (HTML + Plain)
 */
function yc_sendMailCode(string $email, string $name, string $code, string $expiresAtDb, int $ttlSeconds = 60): bool {
  $fromEmail = EB_FROM_EMAIL;
  $fromName  = EB_FROM_NAME;

  $subject = "كود التحقق لإتمام: تأكيد البريد";
  if (function_exists('mb_encode_mimeheader')) {
    $subject = mb_encode_mimeheader($subject, "UTF-8", "B", "\r\n");
  }

  $plain = yc_build_email_plain($name, $code, $expiresAtDb, $ttlSeconds);
  $html  = yc_build_email_html($email, $name, $code, $expiresAtDb, $ttlSeconds);

  $boundary = "b1_" . bin2hex(random_bytes(10));

  @ini_set('sendmail_from', $fromEmail);

  $headers = [];
  $headers[] = "MIME-Version: 1.0";
  $headers[] = "From: {$fromName} <{$fromEmail}>";
  $headers[] = "Reply-To: {$fromEmail}";
  $headers[] = "X-Mailer: PHP/" . phpversion();
  $headers[] = "Content-Type: multipart/alternative; boundary=\"{$boundary}\"";
  $headersStr = implode("\r\n", $headers);

  $message  = "";
  $message .= "--{$boundary}\r\n";
  $message .= "Content-Type: text/plain; charset=UTF-8\r\n";
  $message .= "Content-Transfer-Encoding: 8bit\r\n\r\n";
  $message .= $plain . "\r\n\r\n";

  $message .= "--{$boundary}\r\n";
  $message .= "Content-Type: text/html; charset=UTF-8\r\n";
  $message .= "Content-Transfer-Encoding: 8bit\r\n\r\n";
  $message .= $html . "\r\n\r\n";

  $message .= "--{$boundary}--\r\n";

  // envelope sender (قد يفشل في بعض الاستضافات) + fallback
  $params = "-f{$fromEmail}";
  $ok = @mail($email, $subject, $message, $headersStr, $params);
  if (!$ok) {
    $ok = @mail($email, $subject, $message, $headersStr);
  }

  return (bool)$ok;
}

/**
 * المصدر الواحد لإرسال/إعادة إرسال كود البريد
 * @return array ['success'=>bool,'message'=>string,'status'=>int,'extra'=>array]
 *
 * ملاحظة TX:
 * - إن بدأنا Transaction داخل الدالة: سنعمل commit/rollback.
 * - إن كانت هناك Transaction خارجية: لا نعمل commit/rollback.
 */
function yc_email_request_send_code(PDO $conn, int $userId, string $email, string $name, int $ttlSeconds = 60, bool $applyRateLimit = true): array {
  $startedTx = false;

  try {
    // Rate limit: آخر إرسال خلال ttl
    if ($applyRateLimit) {
      $lastQ = $conn->prepare("
        SELECT created_at
        FROM email_code
        WHERE email = ? AND user_id = ?
        ORDER BY id DESC
        LIMIT 1
      ");
      $lastQ->execute([$email, $userId]);
      $last = $lastQ->fetch(PDO::FETCH_ASSOC);

      if ($last && !empty($last['created_at'])) {
        $lastTs = strtotime((string)$last['created_at']) ?: 0;
        if ($lastTs > 0) {
          $diff = time() - $lastTs;
          if ($diff < $ttlSeconds) {
            $remain = $ttlSeconds - $diff;
            return [
              'success' => false,
              'message' => "تم إرسال رمز مؤخراً. حاول بعد {$remain} ثانية.",
              'status'  => 429,
              'extra'   => ['retry_after' => $remain]
            ];
          }
        }
      }
    }

    $code        = (string) random_int(100000, 999999);
    $expiresAtDb = date('Y-m-d H:i:s', time() + $ttlSeconds);

    if (!$conn->inTransaction()) {
      $conn->beginTransaction();
      $startedTx = true;
    }

    // إغلاق الأكواد السابقة غير المستخدمة
    $conn->prepare("UPDATE email_code SET used = 1 WHERE user_id = ? AND used = 0")
         ->execute([$userId]);

    // إدخال كود جديد
    $ins = $conn->prepare("
      INSERT INTO email_code (email, user_id, code, used, expires_at, created_at)
      VALUES (?, ?, ?, 0, ?, NOW())
    ");
    $ins->execute([$email, $userId, $code, $expiresAtDb]);

    // إرسال البريد
    $sent = yc_sendMailCode($email, $name, $code, $expiresAtDb, $ttlSeconds);
    if (!$sent) {
      if ($startedTx && $conn->inTransaction()) $conn->rollBack();
      return [
        'success' => false,
        'message' => 'تم إنشاء رمز التفعيل ولكن تعذّر إرسال البريد. حاول لاحقاً.',
        'status'  => 500,
        'extra'   => []
      ];
    }

    if ($startedTx && $conn->inTransaction()) $conn->commit();

    return [
      'success' => true,
      'message' => 'تم إرسال رمز التفعيل إلى بريدك الإلكتروني.',
      'status'  => 200,
      'extra'   => [
        'email'      => $email,
        'expires_at' => $expiresAtDb,
        'ttl'        => $ttlSeconds
      ]
    ];

  } catch (Throwable $e) {
    if ($startedTx && $conn->inTransaction()) $conn->rollBack();
    yc_log_error($e);
    return [
      'success' => false,
      'message' => 'خطأ في الخادم',
      'status'  => 500,
      'extra'   => []
    ];
  }
}

/* ================== Endpoint Mode ================== */
if (!defined('YC_EMAIL_REQUEST_LIB')) {

  /* ================== CORS ================== */
  $reqOrigin = $_SERVER['HTTP_ORIGIN'] ?? '';
  if ($reqOrigin === $frontendOrigin) {
    header("Access-Control-Allow-Origin: {$frontendOrigin}");
    header("Vary: Origin");
    header("Access-Control-Allow-Credentials: true");
  }
  header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");
  header("Access-Control-Allow-Methods: POST, OPTIONS");
  header("Access-Control-Max-Age: 600");
  header("Content-Type: application/json; charset=UTF-8");

  $DEBUG_MODE = true;
  if ($DEBUG_MODE) { ini_set('display_errors', 1); error_reporting(E_ALL); }
  else { ini_set('display_errors', 0); error_reporting(0); }

  function respond(bool $success, string $message, array $extra = [], int $status = 200): void {
    http_response_code($status);
    echo json_encode(
      array_merge(['success' => $success, 'message' => $message], $extra),
      JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES
    );
    exit;
  }

  function read_json_input(): array {
    $raw = file_get_contents('php://input') ?: '';
    $data = json_decode($raw, true);
    if (!is_array($data)) {
      respond(false, 'JSON غير صالح.', [], 400);
    }
    return $data;
  }

  if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') {
    http_response_code(204);
    exit;
  }

  if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
    respond(false, 'استخدم POST فقط.', [], 405);
  }

  $input = read_json_input();
  $email = strtolower(trim((string)($input['email'] ?? '')));

  if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
    respond(false, 'بريد غير صالح.', [], 400);
  }

  // DB
  require_once __DIR__ . '/../config.php';
  if (!isset($conn) || !($conn instanceof PDO)) {
    respond(false, 'اتصال قاعدة البيانات غير متوفر.', [], 500);
  }

  try {
    $ttlSeconds = 60;

    // جلب المستخدم
    $u = $conn->prepare("SELECT id, name, is_active FROM users WHERE email = ? LIMIT 1");
    $u->execute([$email]);
    $user = $u->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
      respond(false, 'البريد غير مسجل.', ['exists' => false], 404);
    }

    $userId = (int)($user['id'] ?? 0);
    $name   = (string)($user['name'] ?? '');
    $active = (int)($user['is_active'] ?? 0);

    if ($userId <= 0) {
      respond(false, 'تعذر تحديد المستخدم.', [], 500);
    }

    if ($active === 1) {
      respond(false, 'الحساب مفعل مسبقاً. استخدم تسجيل الدخول.', ['exists' => true, 'is_active' => 1], 409);
    }

    $r = yc_email_request_send_code($conn, $userId, $email, $name, $ttlSeconds, true);
    respond((bool)$r['success'], (string)$r['message'], (array)$r['extra'], (int)$r['status']);

  } catch (Throwable $e) {
    yc_log_error($e);
    $msg = 'خطأ في الخادم';
    if ($DEBUG_MODE) $msg .= ' - ' . $e->getMessage();
    respond(false, $msg, [], 500);
  }
}
