<?php
session_start();

/**
 * password_reset_confirm.php (PDO) - mail() only + reCAPTCHA (Server-side)
 * ✅ يتحقق من الكود ثم يحدّث كلمة المرور في users
 * ✅ يجعل الكود used=1 بعد النجاح أو عند انتهاء الصلاحية
 */

$frontendOrigin = 'https://eazzybit.com';
$reqOrigin = $_SERVER['HTTP_ORIGIN'] ?? '';
if ($reqOrigin === $frontendOrigin) {
  header("Access-Control-Allow-Origin: {$frontendOrigin}");
  header("Vary: Origin");
  header("Access-Control-Allow-Credentials: true");
}
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Max-Age: 600");
header("Content-Type: application/json; charset=UTF-8");

$DEBUG_MODE = true;
if ($DEBUG_MODE) { ini_set('display_errors', 1); error_reporting(E_ALL); }
else { ini_set('display_errors', 0); error_reporting(0); }

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['success' => false, 'message' => 'طريقة الطلب غير مسموحة. استخدم POST فقط.'], JSON_UNESCAPED_UNICODE);
  exit;
}

function respond(bool $success, string $message, array $extra = [], int $status = 200): void {
  http_response_code($status);
  echo json_encode(array_merge(['success' => $success, 'message' => $message], $extra),
    JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES
  );
  exit;
}

function read_json_input(): array {
  $raw = file_get_contents('php://input') ?: '';
  $data = json_decode($raw, true);
  if (!is_array($data)) respond(false, 'البيانات المرسلة غير صالحة', [], 400);
  return $data;
}

function log_error(Throwable $e): void {
  $dir = __DIR__ . '/../../logs';
  if (!is_dir($dir)) @mkdir($dir, 0755, true);
  $file = $dir . '/api_errors_' . date('Y-m-d') . '.log';
  $line = "[" . date('Y-m-d H:i:s') . "] " . $e->getMessage() . " | " . $e->getFile() . ":" . $e->getLine() . PHP_EOL;
  @file_put_contents($file, $line, FILE_APPEND | LOCK_EX);
}

function verify_recaptcha(string $token, string $remoteIp = ''): array {
  if (!defined('RECAPTCHA_SECRET') || !RECAPTCHA_SECRET) {
    return ['ok' => false, 'reason' => 'RECAPTCHA_SECRET_NOT_SET'];
  }

  $url = 'https://www.google.com/recaptcha/api/siteverify';
  $post = ['secret' => RECAPTCHA_SECRET, 'response' => $token];
  if ($remoteIp !== '') $post['remoteip'] = $remoteIp;

  $respRaw = null;

  if (function_exists('curl_init')) {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
      CURLOPT_POST => true,
      CURLOPT_POSTFIELDS => http_build_query($post),
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_CONNECTTIMEOUT => 6,
      CURLOPT_TIMEOUT => 10,
    ]);
    $respRaw = curl_exec($ch);
    curl_close($ch);
  } else {
    $context = stream_context_create([
      'http' => [
        'method'  => 'POST',
        'header'  => "Content-Type: application/x-www-form-urlencoded\r\n",
        'content' => http_build_query($post),
        'timeout' => 10,
      ]
    ]);
    $respRaw = @file_get_contents($url, false, $context);
  }

  if (!$respRaw) return ['ok' => false, 'reason' => 'VERIFY_REQUEST_FAILED'];

  $json = json_decode($respRaw, true);
  if (!is_array($json)) return ['ok' => false, 'reason' => 'VERIFY_BAD_RESPONSE'];

  if (empty($json['success'])) {
    return ['ok' => false, 'reason' => 'VERIFY_NOT_SUCCESS', 'errors' => $json['error-codes'] ?? []];
  }

  if (isset($json['score']) && defined('RECAPTCHA_MIN_SCORE')) {
    $min = (float)RECAPTCHA_MIN_SCORE;
    $score = (float)$json['score'];
    if ($score < $min) return ['ok' => false, 'reason' => 'LOW_SCORE', 'score' => $score];
  }

  return ['ok' => true, 'data' => $json];
}

require_once __DIR__ . '/../config.php';
if (!isset($conn) || !($conn instanceof PDO)) respond(false, 'اتصال قاعدة البيانات غير متوفر.', [], 500);

try {
  $input = read_json_input();

  $email = strtolower(trim((string)($input['email'] ?? '')));
  $code  = trim((string)($input['code'] ?? ''));
  $password = trim((string)($input['password'] ?? ''));
  $confirm  = trim((string)($input['confirm_password'] ?? $input['confirm'] ?? ''));

  $recaptchaToken =
    trim((string)($input['recaptcha_token'] ?? '')) ?:
    trim((string)($input['g_recaptcha_response'] ?? '')) ?:
    trim((string)($input['g-recaptcha-response'] ?? ''));

  if ($email === '' || $code === '' || $password === '') {
    respond(false, 'بعض الحقول ناقصة', [], 400);
  }
  if (!filter_var($email, FILTER_VALIDATE_EMAIL)) respond(false, 'البريد الإلكتروني غير صالح', [], 400);

  if (!preg_match('/^\d{6}$/', $code)) respond(false, 'رمز التحقق يجب أن يكون 6 أرقام', [], 400);

  if (strlen($password) < 6) respond(false, 'كلمة المرور يجب أن تكون 6 أحرف على الأقل', [], 400);
  if ($confirm !== '' && $password !== $confirm) respond(false, 'كلمتا المرور غير متطابقتين', [], 400);

  if ($recaptchaToken === '') respond(false, 'التحقق مطلوب. أعد المحاولة.', ['captcha' => 'required'], 403);

  $remoteIp = $_SERVER['REMOTE_ADDR'] ?? '';
  $cap = verify_recaptcha($recaptchaToken, $remoteIp);
  if (!$cap['ok']) {
    $extra = ['captcha' => 'failed'];
    if ($DEBUG_MODE) $extra['captcha_reason'] = $cap['reason'] ?? 'unknown';
    respond(false, 'فشل التحقق. حاول مرة أخرى.', $extra, 403);
  }

  $conn->beginTransaction();

  // تحقق من وجود المستخدم
  $u = $conn->prepare("SELECT id FROM users WHERE email = ? LIMIT 1");
  $u->execute([$email]);
  $user = $u->fetch(PDO::FETCH_ASSOC);
  if (!$user) {
    $conn->rollBack();
    respond(false, 'البريد غير مسجل لدينا', [], 404);
  }

  // جلب آخر كود صالح وغير مستخدم
  $stmt = $conn->prepare("
    SELECT id, expires_at
    FROM password_reset_code
    WHERE email = ? AND code = ? AND used = 0
    ORDER BY id DESC
    LIMIT 1
  ");
  $stmt->execute([$email, $code]);
  $row = $stmt->fetch(PDO::FETCH_ASSOC);

  if (!$row) {
    $conn->rollBack();
    respond(false, 'الرمز غير صحيح أو تم استخدامه', [], 400);
  }

  // تحقق من الانتهاء
  $expiresAt = (string)$row['expires_at'];
  if (strtotime($expiresAt) < time()) {
    // علّم الرمز منتهي/مغلق
    $conn->prepare("UPDATE password_reset_code SET used = 1 WHERE id = ?")->execute([(int)$row['id']]);
    $conn->commit();
    respond(false, 'انتهت صلاحية الرمز. أعد الإرسال.', ['expired' => true], 400);
  }

  // تحديث كلمة المرور
  $newHash = password_hash($password, PASSWORD_DEFAULT);
  $up = $conn->prepare("UPDATE users SET password = ? WHERE email = ? LIMIT 1");
  $up->execute([$newHash, $email]);

  // إغلاق الكود
  $conn->prepare("UPDATE password_reset_code SET used = 1 WHERE id = ?")
       ->execute([(int)$row['id']]);

  $conn->commit();

  respond(true, 'تم تحديث كلمة المرور بنجاح');

} catch (Throwable $e) {
  if (isset($conn) && $conn instanceof PDO && $conn->inTransaction()) $conn->rollBack();
  log_error($e);
  $msg = 'خطأ في الخادم';
  if ($DEBUG_MODE) $msg .= ' - ' . $e->getMessage();
  respond(false, $msg, [], 500);
}
