<?php
session_start();

/**
 * get_login.php
 * - تسجيل دخول عبر users فقط
 * - إذا is_active=1: يولد user_token ويرجع user_id + email + user_token (+ device)
 * - إذا غير مفعل: يرجع needs_verification=true
 */

$frontendOrigin = 'https://yemencash.net';

/* ================== CORS ================== */
$reqOrigin = $_SERVER['HTTP_ORIGIN'] ?? '';
if ($reqOrigin === $frontendOrigin) {
  header("Access-Control-Allow-Origin: {$frontendOrigin}");
  header("Vary: Origin");
  header("Access-Control-Allow-Credentials: true");
}
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Max-Age: 600");
header("Content-Type: application/json; charset=UTF-8");
/* =================================================== */

$DEBUG_MODE = true;
if ($DEBUG_MODE) { ini_set('display_errors', 1); error_reporting(E_ALL); }
else { ini_set('display_errors', 0); error_reporting(0); }

/* ================== Helpers ================== */
function respond(bool $success, string $message, array $extra = [], int $status = 200): void {
  http_response_code($status);
  echo json_encode(
    array_merge(['success' => $success, 'message' => $message], $extra),
    JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES
  );
  exit;
}

function read_json_input(): array {
  $raw = file_get_contents('php://input') ?: '';
  $data = json_decode($raw, true);
  if (!is_array($data)) respond(false, 'JSON غير صالح.', [], 400);
  return $data;
}

/**
 * Device cookie (ثابت لكل جهاز على نفس الدومين)
 * ملاحظة: لو API على دومين مختلف (.info) فالكوكيز قد تكون طرف ثالث.
 * لضمان الإرسال مع fetch credentials عبر دومين مختلف: SameSite=None + Secure
 */
function getOrCreateDeviceToken(): string {
  $cookieName = 'yc_device';
  if (!empty($_COOKIE[$cookieName])) return (string)$_COOKIE[$cookieName];

  $token = hash('sha256', random_bytes(32));

  $options = [
    'expires'  => time() + (3600 * 24 * 365 * 2),
    'path'     => '/',
    // 'domain' => '.yemencash.net', // فقط لو كان نفس الدومين/ساب دومين
    'secure'   => true,
    'httponly' => true,
    'samesite' => 'None',
  ];

  @setcookie($cookieName, $token, $options);
  return $token;
}

function log_error(Throwable $e): void {
  $dir = __DIR__ . '/../../logs';
  if (!is_dir($dir)) @mkdir($dir, 0755, true);
  $file = $dir . '/api_errors_' . date('Y-m-d') . '.log';
  $line = "[" . date('Y-m-d H:i:s') . "] " . $e->getMessage() . " | " . $e->getFile() . ":" . $e->getLine() . PHP_EOL;
  @file_put_contents($file, $line, FILE_APPEND | LOCK_EX);
}

function columnExists(PDO $conn, string $table, string $column): bool {
  $q = $conn->prepare("
    SELECT 1
    FROM INFORMATION_SCHEMA.COLUMNS
    WHERE TABLE_SCHEMA = DATABASE()
      AND TABLE_NAME = ?
      AND COLUMN_NAME = ?
    LIMIT 1
  ");
  $q->execute([$table, $column]);
  return (bool)$q->fetchColumn();
}
/* =================================================== */

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') respond(false, 'استخدم POST فقط.', [], 405);

$input = read_json_input();

$email    = trim((string)($input['email'] ?? ''));
$password = (string)($input['password'] ?? '');

if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
  respond(false, 'بريد غير صالح.', [], 400);
}
if ($password === '' || mb_strlen($password) < 6) {
  respond(false, 'كلمة المرور غير صالحة.', [], 400);
}

// DB
require_once __DIR__ . '/../config.php';
if (!isset($conn) || !($conn instanceof PDO)) {
  respond(false, 'اتصال قاعدة البيانات غير متوفر.', [], 500);
}

try {
  // جلب المستخدم
  $q = $conn->prepare("
    SELECT id, email, password, is_active
    FROM users
    WHERE email = ?
    LIMIT 1
  ");
  $q->execute([$email]);
  $user = $q->fetch(PDO::FETCH_ASSOC);

  // نفس رسالة الخطأ لعدم كشف وجود البريد
  if (!$user) {
    respond(false, 'بيانات الدخول غير صحيحة.', [], 401);
  }

  $hash = (string)($user['password'] ?? '');
  if ($hash === '' || !password_verify($password, $hash)) {
    respond(false, 'بيانات الدخول غير صحيحة.', [], 401);
  }

  $userId   = (int)$user['id'];
  $isActive = (int)($user['is_active'] ?? 0);

  // غير مفعل -> UI يوجّه إلى email.php
  if ($isActive !== 1) {
    respond(false, 'الحساب غير مُفعّل. سيتم تحويلك لتأكيد البريد.', [
      'needs_verification' => true,
      'email' => $email,
      'user_id' => $userId,
    ], 200);
  }

  // مفعل -> إصدار token + تسجيل جلسة
  $conn->beginTransaction();

  $device    = getOrCreateDeviceToken();
  $userToken = hash('sha256', random_bytes(32));
  $ip        = $_SERVER['REMOTE_ADDR'] ?? null;
  $ua        = $_SERVER['HTTP_USER_AGENT'] ?? '';
  $deviceName  = mb_substr($ua, 0, 150);
  $deviceType  = 'Web Browser';
  $deviceModel = 'Browser';

  // user_devices upsert
  $d = $conn->prepare("
    INSERT INTO user_devices (user_id, device, device_name, device_type, device_model, created_at)
    VALUES (?, ?, ?, ?, ?, NOW())
    ON DUPLICATE KEY UPDATE
      user_id = VALUES(user_id),
      device_name = VALUES(device_name),
      device_type = VALUES(device_type),
      device_model = VALUES(device_model)
  ");
  $d->execute([$userId, $device, $deviceName, $deviceType, $deviceModel]);

  // user_sessions
  $s = $conn->prepare("
    INSERT INTO user_sessions
      (user_id, device, user_token, ip_address, country, governorate, device_name, device_type, device_model, session_start, last_activity_at, session_end, is_online)
    VALUES
      (?, ?, ?, ?, NULL, NULL, ?, ?, ?, NOW(), NOW(), NULL, 1)
  ");
  $s->execute([$userId, $device, $userToken, $ip, $deviceName, $deviceType, $deviceModel]);

  // user_logins
  $l = $conn->prepare("
    INSERT INTO user_logins
      (user_id, ip_address, country, governorate, device_name, device_type, device_model, device, user_token, login_at)
    VALUES
      (?, ?, NULL, NULL, ?, ?, ?, ?, ?, NOW())
  ");
  $l->execute([$userId, (string)$ip, $deviceName, $deviceType, $deviceModel, $device, $userToken]);

  // تحديث users.user_token لو موجود
  if (columnExists($conn, 'users', 'user_token')) {
    $ut = $conn->prepare("UPDATE users SET user_token = ? WHERE id = ? LIMIT 1");
    $ut->execute([$userToken, $userId]);
  }

  $conn->commit();

  $_SESSION['user_id'] = $userId;

  respond(true, 'تم تسجيل الدخول بنجاح.', [
    'user_id'    => $userId,
    'email'      => $email,
    'device'     => $device,
    'user_token' => $userToken,
  ]);

} catch (Throwable $e) {
  if ($conn instanceof PDO && $conn->inTransaction()) $conn->rollBack();
  log_error($e);
  $msg = 'خطأ في الخادم';
  if ($DEBUG_MODE) $msg .= ' - ' . $e->getMessage();
  respond(false, $msg, [], 500);
}
